/* Copyright (C) 2012-2018 RealVNC Ltd. All Rights Reserved.
 */

#ifndef __VNCDEVICEPROVIDER_H__
#define __VNCDEVICEPROVIDER_H__

#include <vncdiscoverysdktypes.h>
#include <vncdeviceprovidercallbacks.h>
#include <stddef.h>

/**
 * \file vncdeviceprovider.h
 *
 * This is the main header file for the VNC Device Provider Interface.
 * To implement the interface you need only include this file.
 *
 * This file defines the VNC Device Provider Interface, the data structure
 * associated with it and the sole entry point for the Device Provider DLL or
 * shared object.
 *
 * \see VNCDeviceProviderInitializeType, VNCDeviceProviderInterface
 * \see vncdeviceprovidercallbacks.h, vncdiscoverysdktypes.h
 */

/**
 * \mainpage VNC Device Provider Native Interface
 *
 * \section section_overview Overview
 *
 * The Discovery SDK is for use within viewer applications and allows
 * discovering of VNC Servers. The Discovery SDK will detect devices that
 * contain a VNC Server and provide a vnccmd string which can be passed to the
 * VNC Viewer SDK to establish a VNC connection to the device. The VNC
 * Discovery SDK can run in parallel with the VNC Viewer SDK and discovering of
 * new servers can happen while the Viewer is connected to an existing server.
 * Several VNC connections can be run in parallel by using several instances of
 * the VNC Viewer SDK. Please consult the VNC Viewer SDK for more information.
 *
 * The Discovery SDK abstracts away from the medium to which the connection to
 * the server is made. It allows plug-ins to be loaded dynamically. A plugin is
 * called a Discoverer and the user of the SDK, the client application, can
 * load a specific plug-in by specifying its type.
 *
 * Pluggable Discoverers provide a means of extending the Discovery SDK to
 * support different types of Discoverers. This allows the applications to
 * discover VNC Servers through different methods, using different
 * transports and discovering on as many interfaces as is required. To find out
 * more about Discoverers, please read "VNC Discovery SDK - Native Discoverer
 * Plug-in Interface".
 *
 * In order to allow Discoverers to share device detection, for example over
 * a certain transport, another type of plug-ins are allowed: Device Providers.
 * The Device Provider would be in a separate library that gets loaded by the
 * Discovery SDK when needed. Because the detection of devices is done from
 * a separate module, access to a device from several Discoverers can be
 * managed and synchronized better. Also, if running on different set-ups, it
 * is possible to re-use the same discoverer, but replace the Device Provider
 * library.
 *
 * An example of a use case of this is if several Discoverers want to know of
 * USB devices, then an USB Device Provider could use the mechanism available
 * on some specific platform to retrieve information about USB devices
 * connected, for example on Linux it would use udev, while on WinCE it would
 * use some custom USB driver. The devices presented to the Discoverers will
 * look the same on both these platforms. And if only one Discoverer needs to
 * access an USB device at any one time, this can be managed by the SDK.
 *
 * This document covers the interface between the VNC Discovery SDK and the
 * plug-in Device Providers. If you want to learn about the Discovery SDK
 * client API please see the "VNC Discovery SDK - Native Interface" document.
 *
 * The Device Provider implementation must implement means of finding devices
 * and their details over a given medium.
 *
 * To implement a Device Provider you must create a DLL shared object that exports
 * an entry point named VNCDeviceProviderInitialize. The entry point must conform
 * to the specification of VNCDeviceProviderInitializeType as described below.
 * The Device Provider must also provide an implementation for some of the methods in
 * the VNCDeviceProviderInterface structure. As a minimum it must implement
 * VNCDeviceProviderSetContext, VNCDeviceProviderStart, VNCDeviceProviderStop
 * and VNCDeviceProviderDestroy.
 *
 * Tha name of the Device Provider DLL or shared object must be
 * vncdeviceprovider-somename.dll in Windows, or
 * libvncdeviceprovider-somename.so in Linux. In that case the type of
 * Discoverer is 'somename' and this will be used by the client application to
 * ask the SDK to load it.
 *
 * The Device Provider communicates back to the application through the
 * VNCDeviceProviderCallbacks.
 *
 * Th SDK can monitor for the Device Provider event handles (file descriptors in
 * Unix) for any activity. When some activity occurs, the device provider will be
 * notified and asked to handle the event. The SDK will retrieve the event
 * handles through VNCDeviceProviderGetEventHandlesEx and notify of an event
 * through VNCDeviceProviderHandleEvent.
 *
 * The Device Provider can also request to be notified of a timeout through
 * VNCDeviceProviderHandleTimeout. The timer is set through
 * VNCDeviceProviderSetTimer.
 *
 * If your platform supports C++ and the STL, then you may wish to use the
 * VNCDeviceProviderBase class as your starting point, rather than implementing
 * this API directly. These classes may be found in the common directory of the
 * Device Provider source tree.
 *
 * \section section_legal Legal information
 *
 * Copyright &copy; 2002-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCDeviceProviderInitializeType, VNCDeviceProviderInterface,
 * VNCDeviceProviderCallbacks
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Called by the SDK to set a context that will be presented in the
 * callbacks.
 *
 * The implementation should associate the requested Device Provider with the
 * given contex. This context should be used henceforth in the callbacks.
 *
 * \param pDeviceProvider The Device Provider that will use the new context.
 * \param pContext The new context.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderCallbacks
 */
typedef void VNCCALL
VNCDeviceProviderSetContext(VNCDeviceProvider* pDeviceProvider, void* vpContext);

/**
 * \brief Sets a specific property on a Device Provider instance (synchronous).
 *
 * If the Device Provider supports different modes of operations, or it has
 * different properties that can be set by the Application, then it should
 * implement this method to allow the Application to set those properties.
 *
 * The Device Provider should be able to detect devices without any settings
 * being necessary. So it should have a default mode of operation. This is to
 * allow simple client applications that don't need to know what settings are
 * necessary to start finding devices.
 *
 * \param pDeviceProvider The Device Provider for which to set the property.
 * \param pProperty The string for the property to set (the key). Ownership of
 *                  the string stays with the SDK.
 * \param pValue The string for the value to set to (or NULL if to unset).
 *               Ownership of the string stays with the SDK.
 *
 * \return VNCDiscoverySDKErrorNone if the property was successfully set,
 *         VNCDiscoverySDKErrorNotSupported if the Device Provider doesn't support
 *         this property, or another error specific for the Device Provider.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDiscoverySDKError, VNCDeviceProviderGetProperty
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDeviceProviderSetProperty(VNCDeviceProvider* pDeviceProvider,
                            const char* pProperty,
                            const char* pValue);

/**
 * \brief Gets a specific property from a Device Provider instance (synchronous).
 *
 * If the Device Provider wants to be able to report to the Application
 * different settings/modes of operations etc, then it has to implement this
 * method to do so.
 *
 * \param pDeviceProvider The Device Provider for which the property is to be retrieved.
 * \param pPropery The string representing the property to be retrieved. The
 *                 ownership of the string stays with the SDK.
 * \param pValue The pointer that needs to be set to the result string. The
 *               ownership of the string is passed to the SDK, therefore the
 *               Device Provider should use VNCDeviceProviderAllocString from the
 *               Support API to allocate the memory for the string. If the
 *               Property is not set the pointer returned is NULL (and the
 *               error is VNCDiscoverySDKErrorNone).
 *
 * \return VNCDiscoverySDKErrorNone if the property was successfully retrieved,
 *         VNCDiscoverySDKErrorNotSupported if the Device Provider doesn't
 *         support this property, or another error specific for the Device
 *         Provider.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDiscoverySDKError, VNCDeviceProviderSetProperty
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDeviceProviderGetProperty(VNCDeviceProvider *pDeviceProvider,
                          const char *pProperty,
                          char **ppValue);

/**
 * \brief Starts the device finding process.
 *
 * Starts the device finding process for a specific Device Provider instance.
 * The implementation should report the outcome of the start call through
 * VNCDeviceProviderNotifyStarted from the callbacks API. This is an
 * asynchronous call, so the Device Provider implementation should return from
 * this method as quickly as possible and it can initiate the notification at
 * a later point.
 *
 * The Device Provider is expected to start looking for devices and any new
 * devices should be returned to the SDK through
 * VNCDeviceProviderNotifyDeviceFound. Devices that exist at the start call
 * should be reported as well, unless they were reported in a previous run of
 * the Device Provider.
 *
 * The Device Provider should also report devices as they are removed. When the
 * Provider is re-started, it should know which devices have been removed since
 * it was last stopped and report them as removed.
 *
 * This call is deprecated as VNCDeviceProviderStartMonitoring and
 * VNCDeviceProviderStartReporting are now available. If the Device Provider
 * implementation may be run with Discovery SDK implementations that were
 * released before version 2.9, then this call should still be implemented as a
 * wrapper to the other two to allow the Device Provider to work with older SDK
 * versions.
 *
 * \param pDeviceProvider The Device Provider instance that is supposed to
 * start.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderNotifyStarted, VNCDeviceProviderNotifyDeviceFound,
 * VNCDeviceProviderNotifyDeviceRemoved, VNCDeviceProviderNotifyStopped
 * \see VNCDeviceProviderStartMonitoring
 * \see VNCDeviceProviderStartReporting
 */
typedef void VNCCALL
VNCDeviceProviderStart(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Called by the SDK to get the event handles of the Device Provider.
 *
 * It is recommended to implement VNCDeviceProviderGetEventHandlesEx instead.
 *
 * \param pDeviceProvider The Device Provider for which to monitor the handles.
 *
 * \return Array of event handles of the Device Provider, the last element in
 * the array must be VNCDeviceProviderNoEventHandle, and no elements before the
 * last one should have that value (in other words it's
 * a VNCDeviceProviderNoEventHandle-terminated array). The array is owned by
 * the Device Provider, so it is the plug-in's responsibility to free any
 * allocated memory. The Device Provider should not change the array outside
 * calls to VNCDeviceProviderGetEventHandles, unless the Device Provider is
 * stopped.
 *
 * \see VNCDeviceProviderGetEventHandlesEx
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderHandleEvent, VNCDeviceProviderSetTimer,
 * VNCDeviceProviderHandleTimeout, VNCDeviceProviderEventHandlesChanged
 */
typedef VNCDiscovererEventHandle *VNCCALL
VNCDeviceProviderGetEventHandles(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Method called by the SDK whenever and event happens on a handle
 * provided by the Device Provider.
 *
 * This method will be called to notify the Device Provider that an event
 * happened on one of the handles provided. If the Device Provider is not
 * providing event handles to the SDK then it doesn't need to implement this
 * method. Otherwise an implementation is necessary.
 *
 * During this call it is safe to change the event handle list, change the
 * value of this handle, or to reset the timer for the handle the Device
 * Provider.
 *
 * \note If you don't stop the timer, or reset it, you will be notified when
 *      the original timeout elapses.
 *
 * \param pDeviceProvider The Device Provider for which the event happened.
 * \param pHandle The handle for which the event occured. Only the value, or
 * the timeout for this handle can be changed from this call, other handles
 * must not change from this callback.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderGetEventHandlesEx, VNCDeviceProviderSetTimer,
 * VNCDeviceProviderHandleTimeout
 */
typedef void VNCCALL
VNCDeviceProviderHandleEvent(VNCDeviceProvider *pDeviceProvider,
    VNCDiscovererEventHandle* pHandle);

/**
 * \brief Method called by the SDK whenever the timeout for the handle
 * requested by the Device Provider is reached. 
 *
 * This method will be called to notify the Device Provider that the requested
 * timeout associated with a handle was reached. 
 *
 * If the Device Provider is not asking for timer access from the SDK then it
 * doesn't need to implement this method. Otherwise an implementation is
 * necessary.
 *
 * During this call it is safe to change the event handle for which the timeout
 * was reached.
 *
 * If the Device Provider wishes to have further timer callbacks it must set the
 * timeout request again.
 *
 * \param pDeviceProvider The Device Provider for which the target time was reached.
 * \param pHandle The handle for which the timer expired. Only the value, or
 * the timeout for this handle can be changed from this call, other handles
 * must not change from this callback.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderGetEventHandlesEx, VNCDeviceProviderSetTimer,
 * VNCDeviceProviderHandleEvent
 */
typedef void VNCCALL
VNCDeviceProviderHandleTimeout(VNCDeviceProvider *pDeviceProvider,
    VNCDiscovererEventHandle* pHandle);

/**
 * \brief Stops the device finding process.
 *
 * Stops the device finding process for a specific Device Provider instance.
 * The implementation should confirm stopping by calling
 * VNCDeviceProviderNotifyStopped, from the Support API. 
 *
 * A Device Provider is expected to be able to re-Start at a later point,
 * therefore it will probably be necessary to take a snapshot of the existing
 * devices.
 *
 * This call is deprecated as VNCDeviceProviderStopMonitoring and
 * VNCDeviceProviderStopReporting are now available. If the Device Provider
 * implementation may be run with Discovery SDK implementations that were
 * released before version 2.9, then this call should still be implemented as a
 * wrapper to the other two to allow the Device Provider to work with older SDK
 * versions.
 * 
 * \param pDeviceProvider The Device Provider instance that needs to stop.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderNotifyStopped, VNCDeviceProviderStart
 * \see VNCDeviceProviderStopMonitoring
 * \see VNCDeviceProviderStopReporting
 */
typedef void VNCCALL
VNCDeviceProviderStop(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Destroys a Device Provider instance.
 *
 * This call destroys a specific Device Provider instance.
 *
 * \param pDeviceProvider The Device Provider that needs to be destroyed.
 *
 * \see VNCDeviceProviderInterface
 */
typedef void VNCCALL
VNCDeviceProviderDestroy(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Asks the Device Provider to provide an array of properties of the
 * device that are relevant to the user.
 *
 * The properties here may be used by the SDK to present to the user if asking
 * them to make a choice of which Discoverer should handle the device. Usually
 * not all the device details would be presented here, just what might be of
 * interest to an end-user.
 *
 * \param pDeviceProvider The Device Provider that found the device.
 * \param pDevice The Device for which to get the properties.
 * \param ppProperties The array of properties of the device. The Device
 * provider should populate this array.
 *
 * \return The size of the property array.
 *
 * \see VNCDeviceInfo, VNCDiscoverySDKKeyValPair,
 * VNCDeviceProviderAllocKeyValPairArray
 */
typedef size_t VNCCALL
VNCDeviceProviderGetDeviceProperties(VNCDeviceProvider *pDeviceProvider,
      const VNCDeviceInfo *pDevice,
      VNCDiscoverySDKKeyValPair **ppProperties);

/**
 * \brief Called by the SDK to get the event handles of the Device Provider.
 *
 * All the handles will be monitored by the SDK and whenever an event happens
 * VNCDeviceProviderHandleEvent will be called. This method will be called each
 * time the Discovery SDK will listen for new event. The event handle list and
 * the handle values should not change between calls to this function. If
 * a value needs to change, or the event handles list needs to be modified,
 * then VNCDeviceProviderEventHandlesChanged should be called by the Device
 * Provider. This will trigger VNCDeviceProviderGetEventHandlesEx to be called
 * again.
 *
 * The event handles will not be monitored when the Device Provider is not
 * started. 
 *
 * If this method is implemented, then VNCDeviceProviderGetEventHandles
 * must not be implemented.
 *
 * \param pDeviceProvider The Device Provider for which to monitor the handles.
 * \param ppWriteNotification On return should point to an array
 * which matches the returned array of handles. Each element in the array will
 * instruct the SDK wether to listen for "write" notifications on the event
 * handle. By default all the handles are treated as "read" handles, but if the
 * corresponding value is set, the handles will be checked for "write" as well.
 * It is possible for the Device Provider not to set the array at all (which
 * means the pointer value will remain NULL). In that case it is assumed that
 * all the handles are to be checked only for read.
 *
 * \return Array of event handles of the Device Provider, the last element in
 * the array must be VNCDeviceProviderNoEventHandle, and no elements before the
 * last one should have that value (in other words it's
 * a VNCDeviceProviderNoEventHandle-terminated array). The array is owned by
 * the Device Provider, so it is the plug-in's responsibility to free any
 * allocated memory. The Device Provider should not change the array outside
 * calls to VNCDeviceProviderGetEventHandlesEx, unless the Device Provider is
 * stopped.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderHandleEvent, VNCDeviceProviderSetTimer,
 * VNCDeviceProviderHandleTimeout, VNCDeviceProviderEventHandlesChanged
 */
typedef VNCDiscovererEventHandle *VNCCALL
VNCDeviceProviderGetEventHandlesEx(VNCDeviceProvider *pDeviceProvider,
    int **ppWriteNotification);

/**
 * \brief Starts monitoring for new devices.
 *
 * Starts the device finding process for a specific Device Provider instance.
 *
 * The implementation must report the outcome of the start call straight away by
 * returning the appropiate error code. VNCDeviceProviderNotifyStarted from the
 * callbacks API must not be used.
 *
 * The Device Provider, once started, it must keep track of devices, but not
 * report anything to the SDK before receiving a call to
 * VNCDeviceProviderStartReporting.
 *
 * If a device is seen as appearing and then disappearing before the reporting
 * is started, then the device should not be reported after
 * VNCDeviceProviderStartReporting is called.
 *
 * This is a new version of VNCDeviceProviderStart. The Device Provider must
 * still implement VNCDeviceProviderStart, as older SDK versions will use that,
 * but it should just be a wrapper around this call and
 * VNCDeviceProviderStartReporting.
 *
 * \param pDeviceProvider The Device Provider instance that is supposed to
 * start monitoring.
 *
 * \return VNCDiscoverySDKErrorNone if monitoring can be started, or an error
 * code if that is not possible.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderNotifyStarted
 * \see VNCDeviceProviderStart
 * \see VNCDeviceProviderStartReporting
 * \see VNCDeviceProviderStopMonitoring
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDeviceProviderStartMonitoring(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Starts reporting of new, or remove devices.
 *
 * Once VNCDeviceProviderNotifyStarted has been called with no error, the Device
 * Provider is expected to start reporting any devices found, or any devices
 * that have been removed.
 *
 * The reporting should resume from where it left when
 * VNCDeviceProviderStopReporting was called last, if re-started. This means
 * that it should report straight away devices that were added/removed while
 * reporting was suspended.
 *
 * \param pDeviceProvider The Device Provider instance that is supposed to
 * start reporting.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderNotifyStarted
 * \see VNCDeviceProviderNotifyDeviceFound
 * \see VNCDeviceProviderNotifyDeviceRemoved
 * \see VNCDeviceProviderStart
 * \see VNCDeviceProviderStartMonitoring
 * \see VNCDeviceProviderStopReporting
 */
typedef void VNCCALL
VNCDeviceProviderStartReporting(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Stops the device finding process.
 *
 * Stops the device finding process for a specific Device Provider instance.
 *
 * If the implementation stops monitoring without being requested, it should
 * keep the error code until this is called and report it then.
 *
 * A Device Provider is expected to be able to re-Start at a later point,
 * therefore it will probably be necessary to take a snapshot of the existing
 * devices.
 *
 * This is a new version of VNCDeviceProviderStop. The Device Provider must
 * still implement VNCDeviceProviderStop, as older SDK versions will use that,
 * but it should just be a wrapper around this call and
 * VNCDeviceProviderStopReporting.
 * 
 * \param pDeviceProvider The Device Provider instance that needs to stop.
 *
 * \return VNCDiscoverySDKErrorNone if the device provider was stopped
 * correctly, or another error code if something went wrong.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderStartMonitoring
 * \see VNCDeviceProviderStopReporting
 * \see VNCDeviceProviderStop
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDeviceProviderStopMonitoring(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief Stops the reporting of new, or remove devices.
 *
 * Once the VNCDeviceProviderNotifyStopped callback is invoked, the Device
 * Provider is expected to stop reporting any devices found, or any devices that
 * have been removed. It must carry on monitoring devices until
 * VNCDeviceProviderStopMonitoring is called.
 *
 * \param pDeviceProvider The Device Provider instance that is supposed to
 * stop reporting.
 *
 * \see VNCDeviceProviderInterface
 * \see VNCDeviceProviderNotifyStopped
 * \see VNCDeviceProviderStopMonitoring
 * \see VNCDeviceProviderStartReporting
 * \see VNCDeviceProviderStop
 */
typedef void VNCCALL
VNCDeviceProviderStopReporting(VNCDeviceProvider *pDeviceProvider);

/**
 * \brief The Interface that the Device Provider should implement. 
 *
 * A Device Provider doesn't need to implement all the members of the structure, 
 * but the minimum requirement is: VNCDeviceProviderSetContext,
 * VNCDeviceProviderStart, VNCDeviceProviderStop, VNCDeviceProviderDestroy.
 *
 * \see VNCDeviceProviderCallbacks, VNCDeviceProviderInitializeType
 */
typedef struct
{
  /**
   * Called by the SDK to set a context that will be presented in the
   */
  VNCDeviceProviderSetContext *setContext;
  /**
   * Sets a specific property on a Device Provider instance (synchronous).
   */
  VNCDeviceProviderSetProperty *setProperty;
  /**
   * Gets a specific property from a Device Provider instance (synchronous).
   */
  VNCDeviceProviderGetProperty *getProperty;
  /**
   * Starts the device finding process.
   */
  VNCDeviceProviderStart *start;
  /**
   * Use VNCDeviceProviderGetEventHandlesEx instead.
   */
  VNCDeviceProviderGetEventHandles *getEventHandles;
  /**
   * Method called by the SDK whenever and event happens on a handle
   */
  VNCDeviceProviderHandleEvent *handleEvent;
  /**
   * Method called by the SDK whenever the timeout for the handle
   */
  VNCDeviceProviderHandleTimeout *handleTimeout;
  /**
   * Stops the device finding process.
   */
  VNCDeviceProviderStop *stop;
  /**
   * Destroys a Device Provider instance.
   */
  VNCDeviceProviderDestroy *destroy;
  /**
   * Asks the Device Provider to provide an array of properties of the device
   * that are relevant to the user.
   */
  VNCDeviceProviderGetDeviceProperties *getDeviceProperties;
  /**
   * Called by the SDK to get the event handles of the Device Provider.
   */
  VNCDeviceProviderGetEventHandlesEx *getEventHandlesEx;
  /**
   * Starts monitoring for new devices.
   */
  VNCDeviceProviderStartMonitoring *startMonitoring;
  /**
   * Starts reporting of new, or remove devices.
   */
  VNCDeviceProviderStartReporting *startReporting;
  /**
   * Stops the device finding process.
   */
  VNCDeviceProviderStopMonitoring *stopMonitoring;
  /**
   * Stops the reporting of new, or remove devices.
   */
  VNCDeviceProviderStopReporting *stopReporting;
} VNCDeviceProviderInterface;

/**
 * \brief The type of the sole entry point to be exported by the Device
 * Provider DLL or shared object.
 *
 * \param pInterface A pointer to the structure that needs to be populated with
 *                   the interface.
 * \param interfaceSize The size of the interface structure.
 * \param pSupportAPI A pointer to the Support API structure provided for the
 *                    Device Provider.
 * \param supportAPISize The size of the support API structure.
 *
 * \return A new instance of the Device Provider, or NULL if an error occured.
 *
 * \see VNCDeviceProviderInterface, VNCDeviceProviderCallbacks
 */
typedef VNCDeviceProvider *VNCCALL
VNCDeviceProviderInitializeType(VNCDeviceProviderInterface *pInterface, 
            size_t interfaceSize,
            VNCDeviceProviderCallbacks *pCallbacks,
            size_t callbacksSize);

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCDEVICEPROVIDER_H__) */

